<?php
// Gemeinsame Helferfunktionen für alle API-Skripte

define('MADMIN_SERVERS_FILE', __DIR__ . '/../data/servers.json');

function madmin_load_servers(): array {
    if (!file_exists(MADMIN_SERVERS_FILE)) {
        return ['servers' => []];
    }
    $raw = file_get_contents(MADMIN_SERVERS_FILE);
    if ($raw === false || trim($raw) === '') {
        return ['servers' => []];
    }
    $data = json_decode($raw, true);
    if (!is_array($data) || !isset($data['servers']) || !is_array($data['servers'])) {
        return ['servers' => []];
    }
    return $data;
}

function madmin_save_servers(array $data): bool {
    $json = json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
    if ($json === false) return false;
    $tmp = MADMIN_SERVERS_FILE . '.tmp';
    if (file_put_contents($tmp, $json) === false) return false;
    return rename($tmp, MADMIN_SERVERS_FILE);
}

function madmin_get_server(string $id): ?array {
    $data = madmin_load_servers();
    foreach ($data['servers'] as $srv) {
        if (isset($srv['id']) && $srv['id'] === $id) {
            return $srv;
        }
    }
    return null;
}

function madmin_ssh_connect(array $server) {
    $host = $server['host'] ?? '';
    $port = (int)($server['port'] ?? 22);
    $user = $server['user'] ?? '';
    $pass = $server['pass'] ?? '';

    if ($host === '' || $user === '' || $pass === '') {
        return null;
    }

    $conn = @ssh2_connect($host, $port);
    if (!$conn) return null;

    if (!@ssh2_auth_password($conn, $user, $pass)) {
        return null;
    }

    return $conn;
}

function madmin_ssh_exec_capture($conn, string $cmd): string {
    $stream = @ssh2_exec($conn, "export PATH=/usr/bin:/bin:/usr/sbin:/sbin; " . $cmd . " 2>&1");
    if (!$stream) return '';
    stream_set_blocking($stream, true);
    $out = stream_get_contents($stream);
    fclose($stream);
    return trim($out);
}

function madmin_json_error(string $msg) {
    header('Content-Type: application/json');
    echo json_encode(['status' => 'error', 'error' => $msg]);
    exit;
}

function madmin_require_server_from_request(): array {
    $id = $_REQUEST['server'] ?? '';
    if ($id === '') {
        madmin_json_error('Missing server parameter');
    }
    $srv = madmin_get_server($id);
    if ($srv === null) {
        madmin_json_error('Unknown server id: ' . $id);
    }
    $conn = madmin_ssh_connect($srv);
    if (!$conn) {
        madmin_json_error('SSH connect/auth failed for server: ' . $id);
    }
    return [$srv, $conn];
}
