<?php
// =========================================================
// Xterm.js WebSocket SSH Bridge
// WebSocket: wss://administration.matdan-srv.com/api/xtermjs.php
// Autor: matdan
// =========================================================

// Wichtig: Dieses Script benötigt die PHP-Extensions:
//   php-ssh2
//   php-sockets
//   php-pcntl
//   php-posix
//
// Installiert per:
// apt install -y php-ssh2 php-cli php-sockets

set_time_limit(0);
ob_implicit_flush();

$address = "0.0.0.0";
$port    = 0; // Port 0 bedeutet: kein TCP-Port, sondern WebSocket über Apache/HTTPS

// We are behind Apache as WebSocket endpoint via mod_proxy_wstunnel
// PHP selbst braucht hier keinen Port öffnen.

// Load Server Config
$serversFile = __DIR__ . "/../data/servers.json";
if (!file_exists($serversFile)) {
    echo "Serverliste fehlt.";
    exit;
}
$servers = json_decode(file_get_contents($serversFile), true);

// Upgrade to WebSocket
if (!function_exists("ws_handshake")) {
    function ws_handshake()
    {
        $key = '';
        foreach (headers_list() as $h) {
            if (stripos($h, "Sec-WebSocket-Key:") !== false) {
                $parts = explode(": ", $h);
                $key = trim($parts[1]);
            }
        }
        if (!$key) {
            http_response_code(400);
            echo "Missing WebSocket Key.";
            exit;
        }

        $acceptKey = base64_encode(sha1($key . "258EAFA5-E914-47DA-95CA-C5AB0DC85B11", true));

        header("HTTP/1.1 101 Switching Protocols");
        header("Upgrade: websocket");
        header("Connection: Upgrade");
        header("Sec-WebSocket-Accept: $acceptKey");
    }
}

ws_handshake();

// WebSocket Frame Helpers
function ws_decode($buffer)
{
    $len = ord($buffer[1]) & 127;
    $maskStart = 2;
    if ($len === 126) {
        $maskStart = 4;
    } elseif ($len === 127) {
        $maskStart = 10;
    }
    $mask = substr($buffer, $maskStart, 4);
    $data = substr($buffer, $maskStart + 4);
    $decoded = '';
    for ($i = 0; $i < strlen($data); $i++) {
        $decoded .= $data[$i] ^ $mask[$i % 4];
    }
    return $decoded;
}

function ws_encode($msg)
{
    $frameHead = [];
    $len = strlen($msg);

    $frameHead[0] = 0x81; // final text frame

    if ($len <= 125) {
        $frameHead[1] = $len;
    } elseif ($len <= 65535) {
        $frameHead[1] = 126;
        $frameHead[2] = ($len >> 8) & 255;
        $frameHead[3] = $len & 255;
    } else {
        $frameHead[1] = 127;
        for ($i = 7; $i >= 0; $i--) {
            $frameHead[$i + 2] = ($len >> (8 * (7 - $i))) & 255;
        }
    }

    $str = "";
    foreach ($frameHead as $val) {
        $str .= chr($val);
    }

    return $str . $msg;
}

// --- CLIENT MESSAGE WAIT ---
$input = fread(STDIN, 2000);
$clientData = ws_decode($input);
$clientMsg = json_decode($clientData, true);

if (!$clientMsg || !isset($clientMsg['server'])) {
    echo ws_encode("Fehler: Kein Server angegeben.");
    exit;
}

$target = $clientMsg['server'];
if (!isset($servers[$target])) {
    echo ws_encode("Fehler: Server nicht gefunden.");
    exit;
}

$host = $servers[$target]['host'];
$user = $servers[$target]['user'];
$pass = $servers[$target]['password'];

// SSH-Verbindung
$conn = ssh2_connect($host, 22);
if (!$conn) {
    echo ws_encode("SSH Fehler: Verbindung fehlgeschlagen");
    exit;
}

if (!ssh2_auth_password($conn, $user, $pass)) {
    echo ws_encode("SSH Fehler: Login fehlgeschlagen");
    exit;
}

$stream = ssh2_shell($conn, "xterm-color", null, 80, 24, SSH2_TERM_UNIT_CHARS);
stream_set_blocking($stream, false);

// Terminal-Loop
while (true) {

    // Eingaben vom Client → SSH
    $r = [$stdin = STDIN];
    $w = $e = null;
    if (stream_select($r, $w, $e, 0, 20000)) {
        $input = fread(STDIN, 2000);
        if ($input) {
            $decoded = ws_decode($input);
            fwrite($stream, $decoded);
        }
    }

    // Ausgabe von SSH → Client
    $sshOut = stream_get_contents($stream);
    if ($sshOut !== false && strlen($sshOut) > 0) {
        echo ws_encode($sshOut);
    }

    usleep(20000);
}

?>
