<?php
// files.php – Dateiverwaltung via SSH SFTP: list, stat, download, upload, mkdir, delete, rename, move
// Download liefert binäre Daten, alle anderen Actions JSON.
// Voraussetzung: config.php liefert $config['servers'][<key>] mit host, user, pass

$config = require __DIR__ . '/config.php';

// Für JSON-Antworten per Default
$action = $_REQUEST['action'] ?? '';
if ($action !== 'download') {
  header('Content-Type: application/json');
}

$serverKey = $_REQUEST['server'] ?? '';
$path      = $_REQUEST['path'] ?? '';
$target    = $_REQUEST['target'] ?? '';
$newname   = $_REQUEST['newname'] ?? '';

if (!isset($config['servers'][$serverKey])) {
  echo json_encode(['status'=>'error','error'=>'Unknown server']); exit;
}
$server = $config['servers'][$serverKey];

$conn = @ssh2_connect($server['host'], 22);
if (!$conn) { echo json_encode(['status'=>'error','error'=>'SSH connection failed']); exit; }
if (!@ssh2_auth_password($conn, $server['user'], $server['pass'])) {
  echo json_encode(['status'=>'error','error'=>'Authentication failed']); exit;
}
$sftp = @ssh2_sftp($conn);
if (!$sftp) { echo json_encode(['status'=>'error','error'=>'SFTP init failed']); exit; }

// Hilfsfunktionen
function sftp_realpath($sftp, $p) {
  // Wenn Pfad leer ist, auf $HOME setzen
  if ($p === '' || $p === '/') return '.';
  return $p;
}
function sftp_stat_assoc($sftp, $rpath) {
  $stat = @ssh2_sftp_stat($sftp, $rpath);
  if ($stat === false) return false;
  $type = 'file';
  $mode = $stat['mode'] ?? 0;
  if (($mode & 0x4000) === 0x4000) $type = 'dir';
  if (($mode & 0xA000) === 0xA000) $type = 'link';
  return [
    'size' => $stat['size'] ?? 0,
    'mtime'=> isset($stat['mtime']) ? date('Y-m-d H:i:s', $stat['mtime']) : null,
    'type' => $type,
    'mode' => $mode
  ];
}
function sftp_join($base, $name) {
  if ($base === '' || $base === '/') return '/' . ltrim($name, '/');
  return rtrim($base, '/') . '/' . ltrim($name, '/');
}

// Actions
switch ($action) {

  // LIST DIRECTORY
  case 'list': {
    $rp = sftp_realpath($sftp, $path);
    $uri = "ssh2.sftp://" . intval($sftp) . $rp;
    $dir = @opendir($uri);
    if (!$dir) { echo json_encode(['status'=>'error','error'=>'Cannot open directory','path'=>$rp]); exit; }
    $items = [];
    while (($file = readdir($dir)) !== false) {
      if ($file === '.' || $file === '..') continue;
      $fullRemotePath = $rp === '.' ? $file : sftp_join($rp, $file);
      $st = sftp_stat_assoc($sftp, $fullRemotePath);
      if ($st === false) continue;
      $items[] = [
        'name' => $file,
        'type' => $st['type'],
        'size' => $st['size'],
        'mtime'=> $st['mtime']
      ];
    }
    closedir($dir);
    // Sort: dirs first, then files, by name
    usort($items, function($a,$b){
      if ($a['type'] === $b['type']) return strcasecmp($a['name'], $b['name']);
      return $a['type']==='dir' ? -1 : 1;
    });
    echo json_encode(['status'=>'ok','path'=>$rp,'count'=>count($items),'items'=>$items]);
    break;
  }

  // STAT
  case 'stat': {
    $rp = sftp_realpath($sftp, $path);
    $st = sftp_stat_assoc($sftp, $rp);
    if ($st === false) { echo json_encode(['status'=>'error','error'=>'stat failed','path'=>$rp]); exit; }
    echo json_encode(['status'=>'ok','path'=>$rp,'stat'=>$st]);
    break;
  }

  // DOWNLOAD (liefert Binärdaten)
  case 'download': {
    $rp = $path !== '' ? $path : '';
    $handle = @fopen("ssh2.sftp://" . intval($sftp) . $rp, 'r');
    if (!$handle) {
      header('Content-Type: application/json');
      echo json_encode(['status'=>'error','error'=>'Cannot open remote file','path'=>$rp]); exit;
    }
    $basename = basename($rp);
    header('Content-Type: application/octet-stream');
    header('Content-Disposition: attachment; filename="'.rawbasename($basename).'"');
    header('Cache-Control: no-cache');
    while (!feof($handle)) {
      echo fread($handle, 8192);
      flush();
    }
    fclose($handle);
    exit; // wichtig, keine weitere Ausgabe
  }

  // UPLOAD (multipart form-data)
  case 'upload': {
    if (!isset($_FILES['file'])) {
      echo json_encode(['status'=>'error','error'=>'No file uploaded']); exit;
    }
    $rp = sftp_realpath($sftp, $path);
    $dest = sftp_join($rp, $_FILES['file']['name']);
    $ok = @ssh2_scp_send($conn, $_FILES['file']['tmp_name'], $dest, 0644);
    if (!$ok) { echo json_encode(['status'=>'error','error'=>'Upload failed']); exit; }
    echo json_encode(['status'=>'ok','path'=>$dest,'size'=>$_FILES['file']['size']]);
    break;
  }

  // MKDIR
  case 'mkdir': {
    $rp = sftp_realpath($sftp, $path);
    $ok = @ssh2_sftp_mkdir($sftp, $rp, 0755, true);
    if (!$ok) { echo json_encode(['status'=>'error','error'=>'mkdir failed','path'=>$rp]); exit; }
    echo json_encode(['status'=>'ok','path'=>$rp]);
    break;
  }

  // DELETE file or dir (dir rekursiv via rm -rf)
  case 'delete': {
    $rp = sftp_realpath($sftp, $path);
    $st = sftp_stat_assoc($sftp, $rp);
    if ($st === false) { echo json_encode(['status'=>'error','error'=>'not found','path'=>$rp]); exit; }
    if ($st['type'] === 'dir') {
      // vorsichtig: rekursiv
      $cmd = "bash -lc " . escapeshellarg("rm -rf -- " . $rp);
      $out = ssh2_exec($conn, $cmd);
      if (!$out) { echo json_encode(['status'=>'error','error'=>'rm -rf failed']); exit; }
      stream_set_blocking($out, true); stream_get_contents($out); fclose($out);
      echo json_encode(['status'=>'ok','deleted'=>'dir','path'=>$rp]);
    } else {
      $ok = @ssh2_sftp_unlink($sftp, $rp);
      if (!$ok) { echo json_encode(['status'=>'error','error'=>'unlink failed','path'=>$rp]); exit; }
      echo json_encode(['status'=>'ok','deleted'=>'file','path'=>$rp]);
    }
    break;
  }

  // RENAME
  case 'rename': {
    $rp = sftp_realpath($sftp, $path);
    if ($newname === '') { echo json_encode(['status'=>'error','error'=>'Missing newname']); exit; }
    $dir = preg_replace('~/[^/]+$~', '', $rp);
    $dest = ($dir===''?'/':$dir) . '/' . ltrim($newname, '/');
    $ok = @ssh2_sftp_rename($sftp, $rp, $dest);
    if (!$ok) { echo json_encode(['status'=>'error','error'=>'rename failed']); exit; }
    echo json_encode(['status'=>'ok','from'=>$rp,'to'=>$dest]);
    break;
  }

  // MOVE
  case 'move': {
    $rp = sftp_realpath($sftp, $path);
    $t  = sftp_realpath($sftp, $target);
    // Ziel ist ein Verzeichnis
    $base = basename($rp);
    $dest = rtrim($t,'/') . '/' . $base;
    $ok = @ssh2_sftp_rename($sftp, $rp, $dest);
    if (!$ok) { echo json_encode(['status'=>'error','error'=>'move failed','from'=>$rp,'to'=>$dest]); exit; }
    echo json_encode(['status'=>'ok','from'=>$rp,'to'=>$dest]);
    break;
  }

  // VIEW text – liest bis 200KB
  case 'view': {
    $rp = sftp_realpath($sftp, $path);
    $fh = @fopen("ssh2.sftp://" . intval($sftp) . $rp, 'r');
    if (!$fh) { echo json_encode(['status'=>'error','error'=>'open failed']); exit; }
    $max = 200*1024; $read = 0; $buf = '';
    while (!feof($fh) && $read < $max) {
      $chunk = fread($fh, min(8192, $max-$read));
      if ($chunk === false) break;
      $buf .= $chunk; $read += strlen($chunk);
    }
    fclose($fh);
    echo json_encode(['status'=>'ok','path'=>$rp,'size'=>$read,'content'=>$buf]);
    break;
  }

  default:
    echo json_encode(['status'=>'error','error'=>'Unknown action']);
}
